% simulateddata_SV.m
% 
% simulates data for model with stochastic volatility
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
% INPUTS:   GLOBALS = structure with elements describing the model and 
%                       options for generating the results, has elements
%                       ny = number of nonpredetermined variables
%                       nx = number of predetermined variables
%                       ne = number of exogenous shocks
%                       ns = number of regimes
%                       nY = number of observables
%                       T  = length of simulation after an initial burn-in
%                       burn = length of the burn-in period
%           Pss     = steady state of transition matrix across regimes
%           xss     = steady state of predetermined variables
%           yss     = steady state of nonpredetermined variables
%           H       = first order transition matrix of x
%           G       = first order transition matrix of y
%           H2      = second order transition matrix of x
%           G2      = second order transition matrix of y
%           obsindex = indexes for observable variables, where
%                       the i-th observed variable is element obsindex(i)
%                       of the stacked variables [x ; y]    
%                        
% OUTPUTS:  simdataout	= (nY x T)-matrix where element [i,t] is the
%                       simulated i-th variable at time t
%           regimesout  = T-vector where element [t] is the simulated
%                       regime in period t
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
function [simdataout,regimesout] = ...
    simulateddata_SV(GLOBALS,THETA,Pss,xss,yss,H,G,H2,G2)

% -- Setup -- %
ny = GLOBALS.ny;
nx = GLOBALS.nx;
ne = GLOBALS.ne;
T  = GLOBALS.simlength;
burn = GLOBALS.simburn;


% -- Parameters -- %
try
    ggamma00        = log(GLOBALS.parameters.p01/(1-GLOBALS.parameters.p01));
    ggamma10        = log(GLOBALS.parameters.p10/(1-GLOBALS.parameters.p10));
catch
    ggamma00        = log(THETA(GLOBALS.parm_ind.p01)/(1-THETA(GLOBALS.parm_ind.p01)));
    ggamma10        = log(THETA(GLOBALS.parm_ind.p10)/(1-THETA(GLOBALS.parm_ind.p10)));
end
   
try
    ggamma01        = exp(THETA(GLOBALS.parm_ind.loggamma01));
    ggamma11        = exp(THETA(GLOBALS.parm_ind.loggamma11)); 
catch
    ggamma01        = exp(GLOBALS.parameters.loggamma01);
    ggamma11        = exp(GLOBALS.parameters.loggamma11);
end


psigma_ll   = THETA(GLOBALS.parm_ind.psigma_ll);
psigma_hh   = THETA(GLOBALS.parm_ind.psigma_hh);
Q           = [psigma_ll 1-psigma_ll; 1-psigma_hh psigma_hh];


% --  Initializations -- %
xsim1 = zeros(nx,T+burn);   % first order effects for x
xsim2 = zeros(nx,T+burn);   % second order effects for x
xsim = zeros(nx,T+burn);    % total effects for x
ysim1 = zeros(ny,T+burn);   % first order effects for y
ysim2 = zeros(ny,T+burn);   % second order effects for y
ysim = zeros(ny,T+burn);    % total effects for y
regimes = zeros(1,T+burn);

% -- Simulate Shocks -- %
eps = randn(ne,T+burn);     % simulated shocks
P = Pss;
regimes(1) = 1;
s = 1;

% -- Simulations -- 
for tt = 2:T+burn
    % -- Simulate Regime -- %
    s = sum(rand > cumsum(P(s,:))) + 1;
    regimes(tt) = s;
    
    % -- Predetermined Variables -- %
    xsim1(:,tt) = H{s}*[xsim1(:,tt-1);eps(:,tt);1];    
    xsim2(:,tt) = H{s}*[xsim2(:,tt-1);zeros(ne,1);0] + ...
        0.5*H2{s}*kron([xsim1(:,tt-1);eps(:,tt);1],[xsim1(:,tt-1);eps(:,tt);1]);
    xsim(:,tt) = xss + xsim1(:,tt) + xsim2(:,tt);
    
    % -- Nonpredetermined Variables -- %
    % First Order Effect
    ysim1(:,tt) = G{s}*[xsim1(:,tt-1);eps(:,tt);1];
    ysim2(:,tt) = G{s}*[xsim2(:,tt-1);zeros(ne,1);0] + ...
        0.5*G2{s}*kron([xsim1(:,tt-1);eps(:,tt);1],[xsim1(:,tt-1);eps(:,tt);1]);
    ysim(:,tt) = yss + ysim1(:,tt) + ysim2(:,tt);
    
    % -- Transition Matrix -- %
    bstar       = ysim(GLOBALS.var_ind.y.bstar,tt);
    llambda     = ysim(GLOBALS.var_ind.y.llambda,tt);
    PP(1,1)     = 1 - exp(ggamma00-ggamma01*bstar)/(1+exp(ggamma00-ggamma01*bstar));
    PP(1,2)     = exp(ggamma00-ggamma01*bstar)/(1+exp(ggamma00-ggamma01*bstar));
    PP(2,1)     = exp(ggamma10-ggamma11*llambda)/(1+exp(ggamma10-ggamma11*llambda));
    PP(2,2)     = 1 - exp(ggamma10-ggamma11*llambda)/(1+exp(ggamma10-ggamma11*llambda));
    
    P = kron(PP,Q);
end

% -- Observations -- %
obs = [xsim; ysim];

% -- Output -- %
simdataout = obs(1:nx+ny,burn+1:end)';
regimesout = regimes(1,burn+1:end)';

